/**

 * A jscodeshift transform that detects deprecated imports from @uhg-abyss/* and prints warnings
 */

// Import the list of deprecated components from JSON file
let deprecatedImports;
deprecatedImports = require('./deprecated-imports.json');

module.exports = function (file, api) {
  // Skip non-source files (additional safeguard)
  if (
    file.path.includes('node_modules') ||
    file.path.includes('build/') ||
    file.path.includes('dist/') ||
    file.path.includes('.next/') ||
    file.path.match(/\.min\.js$/)
  ) {
    return null;
  }

  // Wrap the entire operation in a try/catch to avoid crashing on parsing errors
  try {
    const j = api.jscodeshift;
    const root = j(file.source);

    // Track if we found any matching imports
    let foundMatches = false;

    // Find all import declarations
    root.find(j.ImportDeclaration).forEach((path) => {
      const importSource = path.node.source.value;

      // Get the imported component names
      const specifiers = path.node.specifiers || [];

      // Check each imported component against our deprecated list
      specifiers.forEach((specifier) => {
        if (
          specifier.type === 'ImportSpecifier' ||
          specifier.type === 'ImportDefaultSpecifier'
        ) {
          const componentName = specifier.local.name;

          // Check if this component is in our deprecated list
          const deprecatedComponent = deprecatedImports.find((item) => {
            return item.name === componentName;
          });

          if (
            deprecatedComponent &&
            importSource === deprecatedComponent.importPath
          ) {
            foundMatches = true;
            console.log(
              `WARNING: Found deprecated import in ${file.path}: ${componentName} from ${importSource}`
            );
          }
        }
      });
    });

    // Return the original source unchanged
    return foundMatches ? file.source : null; // Returning null means no changes were made
  } catch (error) {
    // If there's a parsing error, log it but don't crash
    console.log(
      `Skipping file ${file.path} due to parsing error: ${error.message}`
    );
    return null;
  }
};
